# MaskHosting Website - Installation Guide

## Quick Start (5 Minutes)

### Step 1: Upload Files
1. Download all PHP files from the outputs folder
2. Upload to your web server via FTP/cPanel File Manager
3. Place files in your public_html or www directory

### Step 2: Configure Site Settings
Edit `config.php` and update these values:

```php
// Replace with your actual URLs
define('SITE_URL', 'https://maskhosting.com');
define('CLIENT_AREA_URL', 'https://my.maskhosting.com');

// Replace with your actual contact information
define('CONTACT_EMAIL', 'support@maskhosting.com');
define('SALES_EMAIL', 'sales@maskhosting.com');
define('CONTACT_PHONE', '+1 (555) 123-4567');

// Add your social media URLs
define('TWITTER_URL', 'https://twitter.com/yourhandle');
define('FACEBOOK_URL', 'https://facebook.com/yourpage');
define('TELEGRAM_URL', 'https://t.me/yourchannel');
```

### Step 3: Set File Permissions
```bash
chmod 644 *.php
chmod 755 .
```

### Step 4: Test Your Site
1. Visit your domain in a browser
2. Test all navigation links
3. Check responsive design on mobile
4. Verify domain search redirects properly

## Detailed Setup

### Requirements
- **Web Server**: Apache or Nginx
- **PHP**: 7.4 or higher
- **HTTPS**: SSL certificate (recommended)
- **Browser**: Modern browser support

### Server Configuration

#### Apache (.htaccess)
Create an `.htaccess` file for clean URLs and security:

```apache
# Force HTTPS
RewriteEngine On
RewriteCond %{HTTPS} off
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]

# Security Headers
<IfModule mod_headers.c>
    Header set X-Frame-Options "SAMEORIGIN"
    Header set X-Content-Type-Options "nosniff"
    Header set X-XSS-Protection "1; mode=block"
    Header set Referrer-Policy "strict-origin-when-cross-origin"
</IfModule>

# Disable Directory Browsing
Options -Indexes

# Cache Control
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
</IfModule>
```

#### Nginx Configuration
Add to your nginx.conf:

```nginx
server {
    listen 80;
    server_name maskhosting.com www.maskhosting.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name maskhosting.com www.maskhosting.com;
    
    root /var/www/html;
    index home.php index.php;
    
    # SSL Configuration
    ssl_certificate /path/to/cert.pem;
    ssl_certificate_key /path/to/key.pem;
    
    # Security Headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;
    
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
        fastcgi_index index.php;
        include fastcgi_params;
    }
}
```

### Contact Form Setup

To make the contact form functional, create `process-contact.php`:

```php
<?php
require_once 'config.php';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = filter_var($_POST['name'], FILTER_SANITIZE_STRING);
    $email = filter_var($_POST['email'], FILTER_SANITIZE_EMAIL);
    $subject = filter_var($_POST['subject'], FILTER_SANITIZE_STRING);
    $message = filter_var($_POST['message'], FILTER_SANITIZE_STRING);
    
    // Validate email
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'Invalid email']);
        exit;
    }
    
    // Send email
    $to = CONTACT_EMAIL;
    $email_subject = "Contact Form: $subject";
    $email_body = "Name: $name\nEmail: $email\n\nMessage:\n$message";
    $headers = "From: $email\r\nReply-To: $email";
    
    if (mail($to, $email_subject, $email_body, $headers)) {
        echo json_encode(['success' => true, 'message' => 'Message sent successfully!']);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to send message']);
    }
}
?>
```

Then update the form action in `contact.php`:
```javascript
document.getElementById('contactForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    fetch('process-contact.php', {
        method: 'POST',
        body: new FormData(this)
    })
    .then(response => response.json())
    .then(data => {
        const formMessage = document.getElementById('formMessage');
        formMessage.style.display = 'block';
        
        if (data.success) {
            formMessage.style.background = 'rgba(0, 255, 136, 0.1)';
            formMessage.style.border = '1px solid var(--primary)';
            formMessage.style.color = 'var(--primary)';
            this.reset();
        } else {
            formMessage.style.background = 'rgba(255, 0, 0, 0.1)';
            formMessage.style.border = '1px solid #ff0000';
            formMessage.style.color = '#ff0000';
        }
        
        formMessage.innerHTML = data.message;
    });
});
```

### Integration with WHMCS/Blesta

If you're using WHMCS or Blesta for your client area:

#### WHMCS Integration
Update URLs in `config.php`:
```php
define('CLIENT_AREA_URL', 'https://my.maskhosting.com');
```

Order links will automatically work with WHMCS cart URLs.

#### Custom Hosting Panel
If using a custom panel, update the order links in each page:
- Change `cart.php?a=add&pid=X` to your order URL format
- Update domain search redirect in JavaScript

### SEO Optimization

#### 1. Create sitemap.xml
```xml
<?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
  <url>
    <loc>https://maskhosting.com/home.php</loc>
    <priority>1.0</priority>
  </url>
  <url>
    <loc>https://maskhosting.com/about.php</loc>
    <priority>0.8</priority>
  </url>
  <url>
    <loc>https://maskhosting.com/pricing.php</loc>
    <priority>0.9</priority>
  </url>
  <!-- Add all other pages -->
</urlset>
```

#### 2. Create robots.txt
```
User-agent: *
Allow: /

Sitemap: https://maskhosting.com/sitemap.xml
```

#### 3. Add Google Analytics
Add before closing `</head>` tag in `header.php`:
```html
<!-- Google Analytics -->
<script async src="https://www.googletagmanager.com/gtag/js?id=GA_MEASUREMENT_ID"></script>
<script>
  window.dataLayer = window.dataLayer || [];
  function gtag(){dataLayer.push(arguments);}
  gtag('js', new Date());
  gtag('config', 'GA_MEASUREMENT_ID');
</script>
```

### Security Checklist

- [ ] SSL certificate installed and working
- [ ] File permissions set correctly (644 for files, 755 for directories)
- [ ] Contact form has CSRF protection
- [ ] All user inputs are sanitized
- [ ] Error reporting disabled in production
- [ ] Regular backups scheduled
- [ ] Server software up to date

### Performance Optimization

#### Enable Gzip Compression
Add to `.htaccess`:
```apache
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript
</IfModule>
```

#### Enable Browser Caching
Already included in the Apache configuration above.

#### Use CDN (Optional)
For better performance, consider using a CDN for:
- Tailwind CSS
- Font Awesome
- Google Fonts

These are already configured via CDN in the template.

### Troubleshooting

#### Issue: Pages not loading
**Solution**: Check that PHP is enabled on your server

#### Issue: Styling not applied
**Solution**: Ensure Tailwind CSS CDN is loading (check browser console)

#### Issue: Contact form not working
**Solution**: Verify PHP mail() function is enabled on server, or implement SMTP

#### Issue: Domain search not redirecting
**Solution**: Check that CLIENT_AREA_URL is correctly set in config.php

### Maintenance

#### Regular Updates
1. **Weekly**: Check for broken links
2. **Monthly**: Review and update pricing if needed
3. **Quarterly**: Update testimonials and statistics
4. **Annually**: Refresh content and design elements

#### Backup Strategy
1. **Daily**: Automated file backups
2. **Weekly**: Database backups (if applicable)
3. **Monthly**: Full site backup to external storage

### Going Live Checklist

- [ ] All contact information updated in config.php
- [ ] Social media links working
- [ ] Contact form tested and working
- [ ] All navigation links tested
- [ ] Mobile responsive design verified
- [ ] SSL certificate installed
- [ ] Google Analytics added
- [ ] Sitemap submitted to search engines
- [ ] 404 error page created
- [ ] Server performance optimized
- [ ] Backups configured

### Support

For customization or technical issues:
1. Check PHP error logs
2. Verify server requirements
3. Test in different browsers
4. Check console for JavaScript errors

---

**Installation Time**: 5-15 minutes
**Difficulty**: Easy
**Technical Knowledge Required**: Basic PHP/FTP

Good luck with your MaskHosting website! 🚀
